function DriverHarmonicFlow

%-------------------------------------------------------------------------
% Example: S1/S2 cases
%
% Reference: 
%   Goldfarb, Donald and Wen, Zaiwen and Yin, Wotao
%   A Curvilinear Search Method for the p-Harmonic Flow on Sphere
%
% Author: Zaiwen Wen, Wotao Yin
%   Version 1.0 .... 2007/11
%-------------------------------------------------------------------------

%-------------------------------------------------------------------------
% !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
% Please run "installmex" for the first usage
%-------------------------------------------------------------------------


clc;
close all;

%------------------------------------------
% Specify an problem to solve

% S1 case
index = 11;

% S2 case
index = 21;    % p = 1
% index = 22;  % p = 1.5
% index = 23;  % p = 2

%
% index = 31;
% index = 32;
% index = 33;
% index = 34;

%------------------------------------------

if index == 11

    noiselevel = 0.2;
    data = DataProb1_S1p2_Dirichlet(noiselevel);
    data.p = 2;
    name = 'Ex1S1p2Dirichlet';
    name = strcat(name,'Noise',num2str(noiselevel));
    data.dt = 1e-3;

    h = data.h;
    % Problem 2
elseif index == 21

    %% S2 test example

    % p-harmonic flow
    p = 1;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = 3*pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.dt = 1e-2;
    data.eps = 1e-6;
    name = 'Ex2S2p1Dirichlet';
elseif index == 22
    p = 3/2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = 3*pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.eps = 1e-6;
    data.dt = 1e-3;
    name = 'Ex2S2p1_5Dirichlet';

elseif index == 23
    p = 2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = 3*pi/2;
%     coeffb = pi/8;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);

    data.dt = 5e-4;
    name = 'Ex2S2p2Dirichlet';

elseif index == 24
    p = 5/2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = 3*pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.dt = 1e-4;
    name = 'Ex2S2p2_5Dirichlet';

    % Problem 2
elseif index == 31

    %% S2 test example

    % p-harmonic flow
    p = 1;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.dt = 1e-3;
    name = 'Ex21S2p1Dirichlet';
elseif index == 32
    p = 3/2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.dt = 1e-3;
    name = 'Ex21S2p1_5Dirichlet';

elseif index == 33
    p = 2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);

    data.dt = 5e-4;
    name = 'Ex21S2p2Dirichlet';

elseif index == 34
    p = 5/2;
    % % grid size
    h = sqrt(2)/2^4;
    % parameters to control the initial solution
    coeffb = pi/2;
    data = DataProb3_S2_Dirichlet(h, p, coeffb);
    data.dt = 1e-3;
    name = 'Ex21S2p2_5Dirichlet';

    % Problem 3
end


% print figures automatically to eps files
% if print_fig_eps = 1, print; otherwise, not
print_fig_eps = 0;

% if check optimality: ||Grad|| <= GradEPS
pars.GradEPS = 1e-5;

% for mix ls-BB
pars.n_ls_monotone = 20;

% maximum number of iterations
% pars.maxiter = 2000;
pars.maxiter = 10000;
% pars.n_ls_monotone = 50;
% pars.n_ls_monotone = ceil(0.1*pars.maxiter);

% print out debug information or not
pars.debug = 0;


% % generate data for ampl file
% GenerateDataAmplS2(data);
% return;

%% set up the algorithm
% function needed by the gradient flow algorithm
pars.EnergyFun  = 'EnergyHarmonicDiscrete';
pars.DiffCommon = 'DiffCommonDiscrete';
pars.UpdateUScheme = 'UpdateUSchemeImplicit1Discrete';
pars.CopyUtToU = 'CopyUtToU';

% additional function needed by ls-BB algorithm
pars.CopyUInfoToTmp ='CopyUInfoToTmpStorage';

% function needed by line search algorithms
pars.EnergyGrad = 'EnergyGradientHarmonicDiscrete';
pars.Energy_Deriv = 'EnergyDirectionDerivHarmonicDiscrete';

% append name
name = strcat(name,'h',num2str(h, '%2.1e'), 'Iter', ...
    num2str(pars.maxiter,'%2.1e'),'Opt',num2str(pars.GradEPS, '%2.1e'));

perf_fig = figure(1);
title(strcat('Exact Solution,' , ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'ExactSoluP','.eps'))
end

perf_fig = figure(2); nfig = 2;
title(strcat('noise Solution,' , ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'NoiseSoluP','.eps'))
end


%% Gradient flow algorithm
dt = data.dt;

tic
[data_GradF, E_GradF, Diff_GradF, output_GradF] = GradientflowHarmonic(data, pars);
output_GradF.cpu = toc;

fprintf('\nRunning time of gradient flow with step size %f is %f \n', dt, output_GradF.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_GradF.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_GradF.GradNorm);

showquiver(data_GradF.u, data_GradF.v,1);

nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('Gradient Flow,' , ' p = ' , num2str(data.p), ' ,dt = ', num2str(dt)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'GradFlowFixdt', num2str(dt) ,'.eps'))
end

%% line search - BB algorithm

pars.debug = 0;
pars.bb = 3;
tic

[data_bb, E_bb, Diff_bb, dt_hist_bb, output_bb]= Harmonic_ls_BB_mix(data, pars);
%[data_bb, E_bb, Diff_bb, dt_hist_bb, output_bb]= Harmonic_ls_GBB_mix(data, pars);
pars.debug = 0;


output_bb.cpu = toc;
fprintf('\nRunning time of bb: %f \n',  output_bb.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_bb.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_bb.GradNorm);
showquiver(data_bb.u, data_bb.v,1);


nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('ls-BB,' ,  ' p = ' , num2str(data.p) ));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'lsBB', num2str(pars.bb) ,'.eps'))
end



% profile on;
%% line search - csrch algorithm
% pars.linesearch = 'csrch';
tic
[data_ls, E_ls, Diff_ls, dt_hist_ls, output_ls]= Harmonic_ls(data, pars);
output_ls.cpu = toc;
fprintf('\nRunning time of ls: %f \n', output_ls.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_ls.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_ls.GradNorm);

% profile viewer;

showquiver(data_ls.u, data_ls.v,1);

nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('Line search,', ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'ls','.eps'))
end


% write performance to file
% append name
filename = strcat('./result/',name,'Perf','.txt');
fid = fopen(filename, 'w');

fprintf('\n\nDetailed report:');


fprintf('\nResults: Gradient flow:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n',  data.p, output_GradF.cpu, ...
    output_GradF.TotalIter,output_GradF.Total_NFE, output_GradF.Fval, output_GradF.GradNorm);


fprintf('\nResults: BB:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_bb.cpu, ...
    output_bb.TotalIter, output_bb.Total_NFE, output_bb.Fval, output_bb.GradNorm );

fprintf('\nResults: LS:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_ls.cpu, ...
    output_ls.TotalIter, output_ls.Total_NFE, output_ls.Fval, output_ls.GradNorm );


% output to file
fprintf(fid,'\nResults: Gradient flow:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n',  data.p, output_GradF.cpu, ...
    output_GradF.TotalIter,output_GradF.Total_NFE, output_GradF.Fval, output_GradF.GradNorm);


fprintf(fid,'\nResults: BB:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_bb.cpu, ...
    output_bb.TotalIter, output_bb.Total_NFE, output_bb.Fval, output_bb.GradNorm );

fprintf(fid,'\nResults: LS:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_ls.cpu, ...
    output_ls.TotalIter, output_ls.Total_NFE, output_ls.Fval, output_ls.GradNorm );



fclose(fid);

%% plot performance of the algorithms
di1 = 1:output_GradF.TotalIter;
di2 = 1:output_bb.TotalIter;
di3 = 1:output_ls.TotalIter;

nminiter = min([output_GradF.TotalIter, output_bb.TotalIter, output_ls.TotalIter]);
nmaxiter = max([output_GradF.TotalIter, output_bb.TotalIter, output_ls.TotalIter]);

idn = floor(0.04*nminiter):nminiter;
% idn = 5:pars.maxiter+1;
idnD = floor(0.5*nminiter):nminiter;

figure; % subplot(1,2,1);
plot(di1, E_GradF(di1), '-', di2, E_bb(di2), '--', di3, E_ls(di3), '-.','LineWidth',3);
axis([-1 min(nmaxiter+1,2000)  0.90*min([E_GradF(di1) E_bb(di2)  E_ls(di3)]) ...
    1.05*max([E_ls(idn)]) ]);
legend('fix stepsize','ls-BB', 'ls-csrch', 0)
title('Energy');
if print_fig_eps == 1
    nfig = nfig + 1;
    perf_fig = figure(nfig);
    print(perf_fig , '-depsc2', strcat('.\result\',name,'EnergylsBBHist','.eps'))
end



figure; % subplot(1,2,1);
plot(di1, E_GradF(di1), '-', di2, E_bb(di2), '--', di3, E_ls(di3), '-.','LineWidth',3);

axis([-1 nminiter+50  0.99*min([E_GradF(di1) E_bb(di2)  E_ls(di3)]) ...
    1.05*max([E_ls(idnD)]) ]);
legend('fix stepsize','ls-BB', 'ls-csrch', 0)
title('Energy');
if print_fig_eps == 1
    nfig = nfig + 1;
    perf_fig = figure(nfig);
    print(perf_fig , '-depsc2', strcat('.\result\',name,'EnergylsBBHistMagnify','.eps'))
end


