function data = DiffCommonDiscrete(data)
%-------------------------------------------------------------------------
% Preparing data structure 
%
% Reference: 
%   Goldfarb, Donald and Wen, Zaiwen and Yin, Wotao
%   A Curvilinear Search Method for the p-Harmonic Flow on Sphere
%
% Author: Zaiwen Wen, Wotao Yin
%   Version 1.0 .... 2007/11
%-------------------------------------------------------------------------


%%
% notations:
% finite difference --> FD
% derivatives operator: D
%
% data struture:
% input a structure data
% data. dimS = 2, S2, otherwise, S1
%       p               order of |DU|^p
%       u,v,(w)         variables, include w if S2
%
% output the same structure data by appending:
%
%       ux,uy           first order forward FD, D+u
%       vx,vy
%
% local variables:
%       uxx, uyy        second order FD, D-(D+u)
%
%  Zaiwen Wen,
%  1.0 --> 2007.9.21
%%



%% first order backward FD and mean operator
% two dimension grid: (1:m, 1:n)
%       E = sum _{i=2}^m ( sum _{j=2}^n  ( F_ij )  )
% where
%       F_ij = sqrt( f1x^2 + f1y^2 + f2x^2 + f2y^2 + f3x^2 + f3y^2 + data.eps )^ {p/2}
%

% copy data
u = data.u;     v = data.v;
[m,n]=size(u);
p = data.p;
hx = data.hx;   hy = data.hy;
dimS = data.dimS;

% Call Matlab code for the S1 case
if dimS == 1
    [data.F, data.Diva] = ...
        ComputeDiffCommonDiscreteS1(data);
end
        
% Call c code for the S2 case
if dimS == 2
    w = data.w;
    [data.F, data.Diva, data.Divb, data.Divc] = ...
            ComputeDiffCommonDiscreteS2(p,hx,hy, u, v, w);
end

%-------------------------------------------------------------------------- 
% Matlab code for S1 and S2
%

% %% compute along x direction
% % mean operator, to save computaion, do not division "/2", move it to f1x,
% % same for the step size hx.
% % However, remember to scale back when we compute the "diva, divb, divc"
% 
% zeta_u  = (u(2:m,2:n) + u(1:m-1,2:n));
% % backward finite difference
% delta_u = (u(2:m,2:n) - u(1:m-1,2:n));
% 
% zeta_v  = (v(2:m,2:n) + v(1:m-1,2:n));
% delta_v = (v(2:m,2:n) - v(1:m-1,2:n));
% 
% zeta_x_uvsq = zeta_u.^2 + zeta_v.^2 ;
% if dimS == 1
%     f1x = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_x_uvsq;
%     %     f1x = (2/hx)*(zeta_u .* delta_x_v - zeta_v .* delta_x_u) ./zeta_x_uvsq;
% elseif dimS == 2
%     % copy data
%     w = data.w;
%     zeta_w  = (w(2:m,2:n) + w(1:m-1,2:n));
%     % delta_x_w = (w(2:m-1,2:n-1) - u(1:m-2,2:n-1))/pars.h;
%     delta_w = (w(2:m,2:n) - w(1:m-1,2:n));
% 
%     zeta_x_uvsq = zeta_x_uvsq + zeta_w.^2;
% 
%     f1x = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_x_uvsq;
%     f2x = 2*(zeta_u .* delta_w - zeta_w .* delta_u) ./zeta_x_uvsq;
%     f3x = 2*(zeta_v .* delta_w - zeta_w .* delta_v) ./zeta_x_uvsq;
% 
%     %     f1x = (2/hx)*(zeta_u .* delta_x_v - zeta_v .* delta_x_u) ./zeta_x_uvsq;
%     %     f2x = (2/hx)*(zeta_u .* delta_x_w - zeta_w .* delta_x_u) ./zeta_x_uvsq;
%     %     f3x = (2/hx)*(zeta_v .* delta_x_w - zeta_w .* delta_x_v) ./zeta_x_uvsq;
% end
% 
% % backward finite difference
% 
% 
% %% compute along y direction
% zeta_u  = (u(2:m,2:n) + u(2:m,1:n-1));
% delta_u = (u(2:m,2:n) - u(2:m,1:n-1));
% 
% zeta_v  = (v(2:m,2:n) + v(2:m,1:n-1));
% delta_v = (v(2:m,2:n) - v(2:m,1:n-1));
% 
% zeta_y_uvsq = zeta_u.^2 + zeta_v.^2 ;
% 
% if dimS == 1
%     f1y = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_y_uvsq;
%     %     f1y = (2/hy)*(zeta_u .* delta_y_v - zeta_v .* delta_y_u) ./zeta_y_uvsq;
% elseif dimS == 2
%     
%     zeta_w  = (w(2:m,2:n) + w(2:m,1:n-1));
%     delta_w = (w(2:m,2:n) - w(2:m,1:n-1));
% 
%     zeta_y_uvsq = zeta_y_uvsq + zeta_w.^2;
%     f1y = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_y_uvsq;
%     f2y = 2*(zeta_u .* delta_w - zeta_w .* delta_u) ./zeta_y_uvsq;
%     f3y = 2*(zeta_v .* delta_w - zeta_w .* delta_v) ./zeta_y_uvsq;
% 
%     %     f1y = (2/hy)*(zeta_u .* delta_y_v - zeta_v .* delta_y_u) ./zeta_y_uvsq;
%     %     f2y = (2/hy)*(zeta_u .* delta_y_w - zeta_w .* delta_y_u) ./zeta_y_uvsq;
%     %     f3y = (2/hy)*(zeta_v .* delta_y_w - zeta_w .* delta_y_v) ./zeta_y_uvsq;
% end
% 
% %% compute function value at each grid point
% 
% F = f1x.^2 + f1y.^2;
% 
% if dimS == 2
%     F = F + f2x.^2 + f2y.^2 + f3x.^2 + f3y.^2;
% end
% 
% if p ~= 2
%     % add a small positive number to avoid nondifferentiability
%     if mod(p, 2) ~= 0
%         F = F + data.eps;
%     end
%     %     F = F.^(p/2);
%     data.F = F.^(p/2);
%     
%     % then update F to F.^((p-2)/p);
% %     Fp2 = (data.F)./F;
%     F = (data.F)./F;
% else  % p == 2
%     data.F = F;
% end
% 
% 
% %% compute information related to gradient
% % data.Diva = zeros(m-2,n-2);
% 
% % note: dimension of matrix u is m by n
% %       dimension of matrix: data.fx, fy, zetaxuvsq, zetayuvsq, F
% %       is  m-1 * n-1
% 
% indm = 1:m-2;   indn = 1:n-2;
% 
% % to scale back the matrix, since the mean operator does not divide by 2 yet
% zeta_x_uvsq  = zeta_x_uvsq./4;
% zeta_y_uvsq  = zeta_y_uvsq./4;
% 
% % zeta_x_uvsq  = (hx/4)*zeta_x_uvsq;
% % zeta_y_uvsq  = (hy/4)*zeta_y_uvsq;
% 
% if mod(p,2) ~= 0
%     
% %     f1x = p * Fp2 .* f1x ./zeta_x_uvsq;
% %     f1y = p * Fp2 .* f1y ./zeta_y_uvsq;
% %     if dimS == 2
% %         f2x = p * Fp2 .* f2x ./zeta_x_uvsq;
% %         f3x = p * Fp2 .* f3x ./zeta_x_uvsq;
% % 
% %         f2y = p * Fp2 .* f2y ./zeta_y_uvsq;
% %         f3y = p * Fp2 .* f3y ./zeta_y_uvsq;
% %     end
%     
%     f1x = p * F .* f1x ./zeta_x_uvsq;
%     f1y = p * F .* f1y ./zeta_y_uvsq;
%     if dimS == 2
%         f2x = p * F .* f2x ./zeta_x_uvsq;
%         f3x = p * F .* f3x ./zeta_x_uvsq;
% 
%         f2y = p * F .* f2y ./zeta_y_uvsq;
%         f3y = p * F .* f3y ./zeta_y_uvsq;
%     end
%     
% else
%     f1x = p * f1x ./zeta_x_uvsq;
%     f1y = p * f1y ./zeta_y_uvsq;
%     if dimS == 2
%         f2x = p * f2x ./zeta_x_uvsq;
%         f3x = p * f3x ./zeta_x_uvsq;
% 
%         f2y = p * f2y ./zeta_y_uvsq;
%         f3y = p * f3y ./zeta_y_uvsq;
%     end
% 
% end
% 
% % data.Diva = zeros(m-2,n-2);
% 
% data.Diva = - f1x(indm,indn) + f1x(indm+1,indn) ...
%             - f1y(indm,indn) + f1y(indm,indn+1);
% 
% % f1x = diff(f1x,1,1);    f1x = f1x(indm,indn);
% % f1y = diff(f1y,1,2);    f1y = f1y(indm,indn);
% % data.Diva = f1x + f1y;
%   
% 
% if dimS == 2
% 
%     %     data.Divb = zeros(m-2,n-2);
%     %     data.Divc = zeros(m-2,n-2);
% 
% %     f2x = diff(f2x,1,1);    f2x = f2x(indm,indn);
% %     f2y = diff(f2y,1,2);    f2y = f2y(indm,indn);
% %     data.Divb = -f2x - f2y;
% % 
% %     f3x = diff(f3x,1,1);    f3x = f3x(indm,indn);
% %     f3y = diff(f3y,1,2);    f3y = f3y(indm,indn);
% %     data.Divc = f3x + f3y;
% 
%         data.Divb =   f2x(indm,indn) - f2x(indm+1,indn) ...
%                     + f2y(indm,indn) - f2y(indm,indn+1);
%     
%         data.Divc = - f3x(indm,indn) + f3x(indm+1,indn) ...
%                     - f3y(indm,indn) + f3y(indm,indn+1);
% 
% end



% Matlab for S1 case

function [F, Diva] = ComputeDiffCommonDiscreteS1(data);
        
u = data.u;     v = data.v;
[m,n]=size(u);
p = data.p;
hx = data.hx;   hy = data.hy;
dimS = data.dimS;
%% compute along x direction
% mean operator, to save computaion, do not division "/2", move it to f1x,
% same for the step size hx.
% However, remember to scale back when we compute the "diva, divb, divc"

zeta_u  = (u(2:m,2:n) + u(1:m-1,2:n));
% backward finite difference
delta_u = (u(2:m,2:n) - u(1:m-1,2:n));

zeta_v  = (v(2:m,2:n) + v(1:m-1,2:n));
delta_v = (v(2:m,2:n) - v(1:m-1,2:n));

zeta_x_uvsq = zeta_u.^2 + zeta_v.^2 ;
f1x = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_x_uvsq;
%     f1x = (2/hx)*(zeta_u .* delta_x_v - zeta_v .* delta_x_u) ./zeta_x_uvsq;


% backward finite difference
%% compute along y direction
zeta_u  = (u(2:m,2:n) + u(2:m,1:n-1));
delta_u = (u(2:m,2:n) - u(2:m,1:n-1));

zeta_v  = (v(2:m,2:n) + v(2:m,1:n-1));
delta_v = (v(2:m,2:n) - v(2:m,1:n-1));

zeta_y_uvsq = zeta_u.^2 + zeta_v.^2 ;

f1y = 2*(zeta_u .* delta_v - zeta_v .* delta_u) ./zeta_y_uvsq;
%     f1y = (2/hy)*(zeta_u .* delta_y_v - zeta_v .* delta_y_u) ./zeta_y_uvsq;



%% compute function value at each grid point

F = f1x.^2 + f1y.^2;

if p ~= 2
    % add a small positive number to avoid nondifferentiability
    if mod(p, 2) ~= 0
        F = F + data.eps;
    end
    %     F = F.^(p/2);
    data.F = F.^(p/2);
    
    % then update F to F.^((p-2)/p);
%     Fp2 = (data.F)./F;
    F = (data.F)./F;
else  % p == 2
    data.F = F;
end


%% compute information related to gradient
% data.Diva = zeros(m-2,n-2);

% note: dimension of matrix u is m by n
%       dimension of matrix: data.fx, fy, zetaxuvsq, zetayuvsq, F
%       is  m-1 * n-1

indm = 1:m-2;   indn = 1:n-2;

% to scale back the matrix, since the mean operator does not divide by 2 yet
zeta_x_uvsq  = zeta_x_uvsq./4;
zeta_y_uvsq  = zeta_y_uvsq./4;

if mod(p,2) ~= 0
    f1x = p * F .* f1x ./zeta_x_uvsq;
    f1y = p * F .* f1y ./zeta_y_uvsq;
else
    f1x = p * f1x ./zeta_x_uvsq;
    f1y = p * f1y ./zeta_y_uvsq;
end

Diva = - f1x(indm,indn) + f1x(indm+1,indn) ...
            - f1y(indm,indn) + f1y(indm,indn+1);
