function DriverHarmonicFlowColorImageDenoising
%-------------------------------------------------------------------------
% Example: Color image denoising
%
% Reference: 
%   Goldfarb, Donald and Wen, Zaiwen and Yin, Wotao
%   A Curvilinear Search Method for the p-Harmonic Flow on Sphere
%
% Author: Zaiwen Wen, Wotao Yin
%   Version 1.0 .... 2007/11
%-------------------------------------------------------------------------

%-------------------------------------------------------------------------
% !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
% Please run "installmex" for the first usage
%-------------------------------------------------------------------------

clc;
close all;

% Specify an problem to solve, i.e., call subroutine 'DataPepperDenoising'
%   to construct the example

% select an image
name = 'pepper';
% name = 'clown';
% name = 'fabric';

% if probsize = 's', we only take a part of the image 'onion.png',
%               otherwize, we take the full image 'onion.png'

probsize = 's';  % a part of the image 'onion.png'
probsize = 'b';  % the full image 'onion.png'

% select p
index = 11;  % p = 1
% index = 13;  % p = 2

%------------------------------------------
% A proper stopping rule is important
%------------------------------------------

if index == 11
    noiselevel = 0.5;
    data = DataPepperDenoising(name, noiselevel, probsize);
    data.p = 1;
    data.eps = 1e-5;
    name = strcat(name,'p1Noise',num2str(noiselevel));
    data.dt = 1e-3;
elseif index == 12
    noiselevel = 0.5;
    data = DataPepperDenoising(name, noiselevel, probsize);
    data.p = 1.5;
    name = strcat(name,'p1_5Noise',num2str(noiselevel));
    data.dt = 1e-3;
elseif index == 13
    noiselevel = 0.5;
    data = DataPepperDenoising(name, noiselevel, probsize);
    data.p = 2;
    %     pars.maxiter = 150;
    name = strcat(name,'p2Noise',num2str(noiselevel));
    data.dt = 1e-3;
end


% print figures automatically to eps files
% if print_fig_eps = 1, print; otherwise, not
print_fig_eps = 0;

% check optimality or not
pars.CheckOptimal = 1;

% max number of iterations
pars.maxiter = 500;

% for mix ls-BB
pars.n_ls_monotone = 30;

% print out debug information or not
pars.debug = 0;


pars.GradEPS = data.noiseNorm;

if data.p == 1
    pars.GradEPS = 0.8*data.noiseNorm;
%     pars.GradEPS = sqrt(data.noiseNorm)*5;
end

if data.p == 2
%     pars.GradEPS = 0.1*data.noiseNorm;
    pars.GradEPS = sqrt(data.noiseNorm)/5;
end

% pars.GradEPS = (data.noiseNorm)^( data.p/2 );
fprintf('\nGradEPS: %4.3e\n', pars.GradEPS);


%% set up the algorithm
% function needed by the gradient flow algorithm
pars.EnergyFun  = 'EnergyHarmonicDiscrete';
pars.DiffCommon = 'DiffCommonDiscrete';
pars.UpdateUScheme = 'UpdateUSchemeImplicit1Discrete';
pars.CopyUtToU = 'CopyUtToU';

% additional function needed by ls-BB algorithm
pars.CopyUInfoToTmp ='CopyUInfoToTmpStorage';

% function needed by line search algorithms
pars.EnergyGrad = 'EnergyGradientHarmonicDiscrete';
pars.Energy_Deriv = 'EnergyDirectionDerivHarmonicDiscrete';


perf_fig = figure(1);
title(strcat('Exact Solution,' , ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'ExactSoluP','.eps'))
end

perf_fig = figure(2);     nfig = 2;
title(strcat('noise Solution,' , ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'NoiseSoluP','.eps'))
end

%% Gradient flow algorithm
dt = data.dt;

tic
[data_GradF, E_GradF, Diff_GradF,output_GradF] = GradientflowHarmonic(data, pars);
output_GradF.cpu = toc;
fprintf('\nRunning time of gradient flow with step size %f is %f \n', dt, output_GradF.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_GradF.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_GradF.GradNorm);

PlotRGB(data_GradF.u, data_GradF.v, data_GradF.w, data_GradF.Bright);

nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('Gradient Flow,' , ' p = ' , num2str(data.p), ' ,dt = ', num2str(dt)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'GradFlowFixdt', num2str(dt) ,'.eps'))
end

%% line search - BB algorithm

% pars.debug = 0;

pars.bb = 3;
tic

%     [data_bb, E_bb, Diff_bb, dt_hist_bb, output_bb]= Harmonic_bb(data, pars);
[data_bb, E_bb, Diff_bb, dt_hist_bb, output_bb]= Harmonic_ls_BB_mix(data, pars);
output_bb.cpu = toc;
fprintf('\nRunning time of bb: %f \n',  output_bb.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_bb.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_bb.GradNorm);

PlotRGB(data_bb.u, data_bb.v, data_bb.w, data_bb.Bright);

nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('ls-BB,' , ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'lsBB', num2str(pars.bb) ,'.eps'))
end

% pars.debug = 0;

% profile on;
%% line search - csrch algorithm
pars.linesearch = 'csrch';
tic
[data_ls, E_ls, Diff_ls, dt_hist_ls, output_ls]= Harmonic_ls(data, pars);
output_ls.cpu = toc;
fprintf('\nRunning time of ls: %f \n', output_ls.cpu);
fprintf('Total number of obj fun evluation:, %d \n', output_ls.Total_NFE);
fprintf('Gradient Norm:, %f \n', output_ls.GradNorm);

% profile viewer;
PlotRGB(data_ls.u, data_ls.v, data_ls.w, data_ls.Bright);

nfig = nfig + 1;
perf_fig = figure(nfig);
title(strcat('Line search,', ' p = ' , num2str(data.p)));
if print_fig_eps == 1
    print(perf_fig , '-depsc2', strcat('.\result\',name,'ls','.eps'))
end

% return;


% write performance to file
% append name
filename = strcat('./result/',name,'Perf','.txt');
fid = fopen(filename, 'w');


fprintf('\n\nDetailed report:');

% output to screen
fprintf('\nResults: Gradient flow:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n',  data.p, output_GradF.cpu, ...
    output_GradF.TotalIter,output_GradF.Total_NFE, output_GradF.Fval, output_GradF.GradNorm);


fprintf('\nResults: BB:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_bb.cpu, ...
    output_bb.TotalIter, output_bb.Total_NFE, output_bb.Fval, output_bb.GradNorm );

fprintf('\nResults: LS:');
fprintf('\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf('%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_ls.cpu, ...
    output_ls.TotalIter, output_ls.Total_NFE, output_ls.Fval, output_ls.GradNorm );

% output to file
fprintf(fid,'\nResults: Gradient flow:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n',  data.p, output_GradF.cpu, ...
    output_GradF.TotalIter,output_GradF.Total_NFE, output_GradF.Fval, output_GradF.GradNorm);


fprintf(fid,'\nResults: BB:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_bb.cpu, ...
    output_bb.TotalIter, output_bb.Total_NFE, output_bb.Fval, output_bb.GradNorm );

fprintf(fid,'\nResults: LS:');
fprintf(fid,'\n p \t cpu \t NIT \t NFE \t f(x) \t |g|\n');
fprintf(fid,'%3.2f &\t %f &\t %d &\t %d &\t %4.2e &\t %4.2e  \\\\ \\hline \n', data.p, output_ls.cpu, ...
    output_ls.TotalIter, output_ls.Total_NFE, output_ls.Fval, output_ls.GradNorm );



fclose(fid);

% %% plot performance of the algorithms
% di = 1:pars.maxiter+1;
% 
% idn = floor(0.025*pars.maxiter):pars.maxiter+1;
% % idn = 5:pars.maxiter+1;
% idnD = floor(0.02*pars.maxiter):pars.maxiter+1;
% 
% % idn = 10:pars.maxiter+1;
% % % idn = 5:pars.maxiter+1;
% % idnD = 10:pars.maxiter+1;
% 
% 
% 
% di1 = 1:output_GradF.TotalIter;
% di2 = 1:output_bb.TotalIter;
% di3 = 1:output_ls.TotalIter;
% 
% nminiter = min([output_GradF.TotalIter, output_bb.TotalIter, output_ls.TotalIter]);
% nmaxiter = max([output_GradF.TotalIter, output_bb.TotalIter, output_ls.TotalIter]);
% 
% idn = floor(0.04*nminiter):nminiter;
% % idn = 5:pars.maxiter+1;
% idnD = floor(0.5*nminiter):nminiter;
% 
% figure; % subplot(1,2,1);
% plot(di1, E_GradF(di1), '-', di2, E_bb(di2), '--', di3, E_ls(di3), '-.','LineWidth',3);
% axis([-1 min(nmaxiter+1,2000)  0.90*min([E_GradF(di1) E_bb(di2)  E_ls(di3)]) ...
%     1.05*max([E_ls(idn)]) ]);
% legend('fix stepsize','ls-BB', 'ls-csrch', 0)
% title('Energy');
% if print_fig_eps == 1
%     nfig = nfig + 1;
%     perf_fig = figure(nfig);
%     print(perf_fig , '-depsc2', strcat('.\result\',name,'EnergylsBBHist','.eps'))
% end
% 
% 
% 
% figure; % subplot(1,2,1);
% plot(di1, E_GradF(di1), '-', di2, E_bb(di2), '--', di3, E_ls(di3), '-.','LineWidth',3);
% 
% axis([-1 nminiter+50  0.99*min([E_GradF(di1) E_bb(di2)  E_ls(di3)]) ...
%     1.05*max([E_ls(idnD)]) ]);
% legend('fix stepsize','ls-BB', 'ls-csrch', 0)
% title('Energy');
% if print_fig_eps == 1
%     nfig = nfig + 1;
%     perf_fig = figure(nfig);
%     print(perf_fig , '-depsc2', strcat('.\result\',name,'EnergylsBBHistMagnify','.eps'))
% end
