function [data, E, Diff, dt_hist, output]= Harmonic_bb(data, pars)

%
%   min f(x),   g(x) = grad f(x)
%    s_{k-1} = x_k - x_{k-1};       y_{k-1} = g_k - g_{k-1}
%    BB step 1:     a = s_{k-1}^T y_{k-1} / || y_{k-1} ||^2
%    BB step 2:     a = ||s_{k-1}||^2 / s_{k-1}^T y_{k-1}
%
%   All data are stored in the data structure "data"
%   Input
%       data.u0, v0, w0
%
%   working space:
%       data.u, v, w
%            ut, vt, wt
%            u1, v1, w1
%            Gradu,  Gradv,  Gradw
%            Gradu1, Gradv1, Gradw1
%
%% Size information
% [m,n]=size(u);
% if any([m,n]~=size(v))  error('u and v have different sizes.'); end;

E = zeros(1,pars.maxiter + 1);
Diff = zeros(1,pars.maxiter + 1);
dt_hist = zeros(1,pars.maxiter + 1);

%dt = dt/(h*h);	% we are doing Laplacian


%--------------------------------------------------------------------------
% Compute the first point by exact line search

%% Initial energy and error
% check input data

% the first step of ls-BB is a fixed stepsize
if ~isfield(data, 'dt')
    data.dt = 1e-3;
end

% copy initial solution to the working space
dt = data.dt;
data.u = data.u0;
data.v = data.v0;
if data.dimS == 2
    data.w = data.w0;
end

% set the active grid points ( Dirichlet problem )
data.idx = 2:data.m-1;      idx = data.idx;
data.idy = 2:data.n-1;      idy = data.idy;

% compute finite difference information
data = feval(pars.DiffCommon, data);

% compute objective function value
E(1) = feval(pars.EnergyFun, data);
Diff(1) = feval(pars.DistanceUV, data);
dt_hist(1) = 0;

% compute gradient at current point
data = feval(pars.EnergyGrad, data);

% copy U and grad(Ep(U)) to temporary storage, we need information of the last two steps to decide a stepsize
data = feval(pars.CopyUInfoToTmp, data);


%% now, generate the first trial point by a fix stepsize
% compute the trial point Ut
data = feval(pars.UpdateUScheme, data, dt);

% copy the trial point Ut to U
data = feval(pars.CopyUtToU, data);

% compute error respect to the optimal solution
data = feval(pars.DiffCommon, data);

E(2) = feval(pars.EnergyFun, data);
Diff(2) = feval(pars.DistanceUV, data);
dt_hist(2) = dt;

% compute gradient at current point
data = feval(pars.EnergyGrad, data);


% Print iteration header if debug == 1
if (pars.debug == 1)
    fid = 1;
    fprintf(fid, '\n----------- Projected Gradient Method with BB method ----------- \n');
    fprintf(fid, '%4s \t %10s \t %10s \n', 'Iter', 'dt', 'E(i)');
    fprintf(fid, '%4d \t %e \t %e	\n', 1, 0, E(1));
    fprintf(fid, '%4d \t %e \t %e	\n', 2, dt, E(2));
    %OUTPUT
end

% Ok, begin to find the second point
%--------------------------------------------------------------------------


%% main iteration
for ind = 3 : pars.maxiter+1

    % compute a stepsize using bb method
    s_u_k_1 = data.u(idx,idy) - data.u1(idx,idy);
    s_v_k_1 = data.v(idx,idy) - data.v1(idx,idy);

    y_u_k_1 = data.Gradu - data.Gradu1;
    y_v_k_1 = data.Gradv - data.Gradv1;


    sy = sum(sum(s_u_k_1.*y_u_k_1)) + sum(sum(s_v_k_1.*y_v_k_1));

    if data.dimS == 2
        s_w_k_1 = data.w(idx,idy) - data.w1(idx,idy);
        y_w_k_1 = data.Gradw - data.Gradw1;
        sy = sy + sum(sum(s_w_k_1.*y_w_k_1));
    end

    if (pars.bb == 1) || ( pars.bb == 3 && mod(ind,2) == 0 )

        yy = sum(sum(y_u_k_1.*y_u_k_1 + y_v_k_1.*y_v_k_1));

        if data.dimS == 2
            yy = yy +  sum(sum(y_w_k_1.*y_w_k_1));
        end
        dt = sy / yy;
    end

    if pars.bb == 2 || ( pars.bb == 3 && mod(ind,2) ~= 0 )
        ss = sum(sum(s_u_k_1.*s_u_k_1 + s_v_k_1.*s_v_k_1));

        if data.dimS == 2
            ss = ss +  sum(sum(s_w_k_1.*s_w_k_1));
        end
        dt = ss / sy;

    end

    dt_hist(ind) = dt;

    % copy U and grad(Ep(U)) to temporary storage
    data = feval(pars.CopyUInfoToTmp, data);


    %% now, generate the new trial point
    % compute the trial point Ut
    data = feval(pars.UpdateUScheme, data, dt);

    % copy the trial point Ut to U
    data = feval(pars.CopyUtToU, data);


    %% compute function value and gradient
    % compute error respect to the optimal solution
    data = feval(pars.DiffCommon, data);

    E(ind) = feval(pars.EnergyFun, data);
    Diff(ind) = feval(pars.DistanceUV, data);
    dt_hist(ind) = dt;

    % compute gradient at current point
    data = feval(pars.EnergyGrad, data);

    energy_deriv = feval(pars.Energy_Deriv, 0, data);
    output.GradNorm = abs( energy_deriv );

    if pars.CheckOptimal == 1
        % if satisfy termination rule
        if output.GradNorm <= pars.GradEPS
            output.TotalIter = ind;
            output.Fval = E(ind);
            output.optimal = 1;
            return;
        end
    end

    % output information
    if (pars.debug == 1)
        fprintf(fid, '%4d \t %e \t %e \n', ind, dt, E(ind));
    end



end

output.optimal = 0;
output.Fval = E(pars.maxiter+1);
output.TotalIter = pars.maxiter;
% output.Energy_Deriv = energy_deriv;

% fprintf('Deriv: %f \n', output.Energy_Deriv);


